// Multi-Step Form Logic with GSAP Animations
let currentStep = 1;
const totalSteps = 5; // Changed from 6 to 5
let formData = {};

// Store logo name in memory instead of localStorage
let logoNameFromIndex = "";

// Initialize the form when the page loads
document.addEventListener("DOMContentLoaded", function () {
  initializeForm();
  setupEventListeners();
  updateProgressBar();
  updateNavigationButtons();
  loadSavedData();
});

function initializeForm() {
  // Try to get logo name from URL parameter
  const urlParams = new URLSearchParams(window.location.search);
  const logoName = urlParams.get("name");

  if (logoName) {
    document.getElementById("logoNameStep").value = logoName;
    formData.logoName = logoName;
    logoNameFromIndex = logoName;
  }

  // Initialize GSAP animations
  gsap.set(".step:not(.active)", { opacity: 0, x: 100 });
  gsap.set(".step.active", { opacity: 1, x: 0 });

  // Animate initial step entrance
  gsap.from(".step.active .step-header", {
    duration: 0.8,
    y: 50,
    opacity: 0,
    ease: "power3.out",
    delay: 0.2,
  });

  gsap.from(".step.active .input-group", {
    duration: 0.8,
    y: 30,
    opacity: 0,
    ease: "power3.out",
    delay: 0.4,
  });

  // Animate progress bar entrance
  gsap.from(".progress-container", {
    duration: 1,
    y: -50,
    opacity: 0,
    ease: "power3.out",
  });

  gsap.from(".step-indicator", {
    duration: 0.6,
    scale: 0,
    opacity: 0,
    ease: "back.out(1.7)",
    stagger: 0.1,
    delay: 0.3,
  });
}

function setupEventListeners() {
  // Add enter key support
  document.addEventListener("keydown", function (e) {
    if (e.key === "Enter") {
      e.preventDefault();
      if (currentStep < totalSteps) {
        nextStep();
      } else {
        submitForm();
      }
    }

    // Arrow key navigation
    if (e.key === "ArrowRight" && currentStep < totalSteps) {
      nextStep();
    }
    if (e.key === "ArrowLeft" && currentStep > 1) {
      previousStep();
    }
  });

  // Add input change listeners for real-time validation
  const inputs = document.querySelectorAll("input, select");
  inputs.forEach((input) => {
    input.addEventListener("change", function () {
      saveCurrentStepData();
    });

    input.addEventListener("input", function () {
      saveCurrentStepData();
    });
  });
}

function nextStep() {
  // Validate current step
  if (!validateCurrentStep()) {
    return;
  }

  // Save current step data
  saveCurrentStepData();

  // Show completion animation
  showStepCompletion(currentStep);

  if (currentStep < totalSteps) {
    // Animate current step out
    const currentStepEl = document.getElementById(`step${currentStep}`);
    const nextStepEl = document.getElementById(`step${currentStep + 1}`);

    gsap.to(currentStepEl, {
      duration: 0.5,
      x: -100,
      opacity: 0,
      ease: "power3.in",
      onComplete: function () {
        currentStepEl.classList.remove("active");
        currentStep++;
        nextStepEl.classList.add("active");

        // Ensure UI reflects the new step immediately
        updateStepIndicators();
        updateProgressBar();
        updateNavigationButtons();
        addToHistory(currentStep);

        // Animate next step in
        gsap.fromTo(
          nextStepEl,
          { x: 100, opacity: 0 },
          {
            duration: 0.5,
            x: 0,
            opacity: 1,
            ease: "power3.out",
            onComplete: function () {
              // Animate step content
              gsap.from(`#step${currentStep} .step-header`, {
                duration: 0.6,
                y: 30,
                opacity: 0,
                ease: "power3.out",
              });

              gsap.from(`#step${currentStep} .input-group`, {
                duration: 0.6,
                y: 20,
                opacity: 0,
                ease: "power3.out",
                delay: 0.2,
              });
            },
          }
        );
      },
    });
  }
}

function previousStep() {
  if (currentStep > 1) {
    const currentStepEl = document.getElementById(`step${currentStep}`);
    const prevStepEl = document.getElementById(`step${currentStep - 1}`);

    gsap.to(currentStepEl, {
      duration: 0.5,
      x: 100,
      opacity: 0,
      ease: "power3.in",
      onComplete: function () {
        currentStepEl.classList.remove("active");
        currentStep--;
        prevStepEl.classList.add("active");

        // Ensure UI reflects the new step immediately
        updateStepIndicators();
        updateProgressBar();
        updateNavigationButtons();
        addToHistory(currentStep);

        // Animate previous step in
        gsap.fromTo(
          prevStepEl,
          { x: -100, opacity: 0 },
          {
            duration: 0.5,
            x: 0,
            opacity: 1,
            ease: "power3.out",
            onComplete: function () {
              // Animate step content
              gsap.from(`#step${currentStep} .step-header`, {
                duration: 0.6,
                y: 30,
                opacity: 0,
                ease: "power3.out",
              });

              gsap.from(`#step${currentStep} .input-group`, {
                duration: 0.6,
                y: 20,
                opacity: 0,
                ease: "power3.out",
                delay: 0.2,
              });
            },
          }
        );
      },
    });
  }
}

function validateCurrentStep() {
  const currentStepEl = document.getElementById(`step${currentStep}`);
  const requiredInputs = currentStepEl.querySelectorAll(
    "input[required], select[required]"
  );

  for (let input of requiredInputs) {
    if (!input.value.trim()) {
      // Show error animation
      gsap.fromTo(
        input,
        { x: 0 },
        {
          duration: 0.5,
          x: [10, -10, 10, -10, 0],
          ease: "power2.inOut",
        }
      );

      // Special validation for email step
      if (currentStep === 5) {
        showAlert("Please enter your email address to continue.", "error");
      }

      input.focus();
      return false;
    }
  }

  // Email validation for step 5 (now the combined final step)
  if (currentStep === 5) {
    const emailInput = document.getElementById("email");
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(emailInput.value)) {
      showAlert("Please enter a valid email address.", "error");
      gsap.fromTo(
        emailInput,
        { x: 0 },
        {
          duration: 0.5,
          x: [10, -10, 10, -10, 0],
          ease: "power2.inOut",
        }
      );
      emailInput.focus();
      return false;
    }
  }

  return true;
}

function saveCurrentStepData() {
  switch (currentStep) {
    case 1:
      formData.logoName = document.getElementById("logoNameStep").value;
      break;
    case 2:
      formData.slogan = document.getElementById("slogan").value;
      break;
    case 3:
      formData.industry = document.getElementById("industry").value;
      break;
    case 4:
      formData.keywords = document.getElementById("keywords").value;
      break;
    case 5:
      formData.email = document.getElementById("email").value;
      formData.phone = document.getElementById("phone").value;
      break;
  }

  // Store in memory only (no localStorage)
  console.log("Form data saved:", formData);
}

function showStepCompletion(step) {
  const completionEl = document.getElementById(`completion${step}`);
  if (
    completionEl &&
    (step < 5 || (step === 5 && document.getElementById("email").value))
  ) {
    completionEl.classList.add("show");

    // Animate completion message
    gsap.fromTo(
      completionEl,
      { scale: 0.8, opacity: 0 },
      {
        duration: 0.6,
        scale: 1,
        opacity: 1,
        ease: "back.out(1.7)",
      }
    );

    // Hide after a delay
    setTimeout(() => {
      gsap.to(completionEl, {
        duration: 0.3,
        opacity: 0,
        scale: 0.9,
        ease: "power2.in",
        onComplete: function () {
          completionEl.classList.remove("show");
        },
      });
    }, 1500);
  }
}

function updateStepIndicators() {
  const indicators = document.querySelectorAll(".step-indicator");

  indicators.forEach((indicator, index) => {
    const stepNumber = index + 1;

    if (stepNumber < currentStep) {
      // Completed step
      indicator.classList.add("completed");
      indicator.classList.remove("active");

      gsap.to(indicator, {
        duration: 0.3,
        scale: 1,
        ease: "power2.out",
      });
    } else if (stepNumber === currentStep) {
      // Current step
      indicator.classList.add("active");
      indicator.classList.remove("completed");

      gsap.to(indicator, {
        duration: 0.5,
        scale: 1.1,
        ease: "back.out(1.7)",
      });
    } else {
      // Future step
      indicator.classList.remove("active", "completed");

      gsap.to(indicator, {
        duration: 0.3,
        scale: 1,
        ease: "power2.out",
      });
    }
  });
}

function updateProgressBar() {
  const progressFill = document.getElementById("progressFill");
  if (!progressFill) return;
  const progressPercentage = (currentStep / totalSteps) * 100;

  gsap.to(progressFill, {
    duration: 0.8,
    width: `${progressPercentage}%`,
    ease: "power2.out",
  });
}

function updateNavigationButtons() {
  const prevBtn = document.getElementById("prevBtn");
  const nextBtn = document.getElementById("nextBtn");
  if (currentStep > 1) {
    prevBtn.classList.add("show");
    prevBtn.style.visibility = "visible";
    prevBtn.style.opacity = "1";
  } else {
    prevBtn.classList.remove("show");
    prevBtn.style.visibility = "hidden";
    prevBtn.style.opacity = "0";
  }
  if (currentStep === totalSteps) {
    nextBtn.style.display = "none";
  } else {
    nextBtn.style.display = "flex";
  }
}

function showAlert(message, type = "info") {
  // Create alert element
  const alert = document.createElement("div");
  alert.className = `alert alert-${type}`;
  alert.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 1rem 1.5rem;
        background: ${
          type === "error" ? "var(--error-color)" : "var(--primary-color)"
        };
        color: white;
        border-radius: var(--radius-md);
        box-shadow: var(--shadow-lg);
        z-index: 1000;
        max-width: 400px;
        font-weight: 600;
        transform: translateX(100%);
    `;
  alert.textContent = message;

  document.body.appendChild(alert);

  // Animate in
  gsap.to(alert, {
    duration: 0.5,
    x: 0,
    ease: "power3.out",
  });

  // Auto remove after 3 seconds
  setTimeout(() => {
    gsap.to(alert, {
      duration: 0.5,
      x: "100%",
      ease: "power3.in",
      onComplete: function () {
        if (alert.parentNode) {
          alert.parentNode.removeChild(alert);
        }
      },
    });
  }, 3000);
}

function submitForm() {
  // Validate final step
  if (!validateCurrentStep()) {
    return;
  }

  // Save final step data
  saveCurrentStepData();

  // Show loading state
  const submitBtn = document.querySelector(".final-step-actions .btn-primary");
  const originalText = submitBtn.innerHTML;
  submitBtn.innerHTML = '<div class="loading"></div> Creating Your Logo...';
  submitBtn.disabled = true;

  // Simulate form submission
  setTimeout(() => {
    // Show success message
    showAlert(
      "Success! Your logo is being created. Check your email in a few minutes.",
      "success"
    );

    // Log form data (in real app, send to server)
    console.log("Form submitted with data:", formData);

    // Reset button
    submitBtn.innerHTML = originalText;
    submitBtn.disabled = false;

    // Redirect or show success page
    setTimeout(() => {
      window.location.href = '/thankyou'
    }, 2000);
  }, 2000);
}

// Utility function to load saved data on page refresh
function loadSavedData() {
  // Data is stored in memory only, so this would be empty on refresh
  // In a real app, you'd load from server or pass via URL
  console.log("Current form data:", formData);
  setTimeout(() => {
      nextStep();
    }, 500);
}

// Reset form function
function resetForm() {
  // Clear all form data
  formData = {};
  logoNameFromIndex = "";

  // Reset to first step
  currentStep = 1;

  // Reset all step classes
  document.querySelectorAll(".step").forEach((step, index) => {
    step.classList.remove("active");
    if (index === 0) step.classList.add("active");
  });

  // Reset form fields
  document.querySelectorAll("input, select").forEach((input) => {
    input.value = "";
  });

  // Reset progress and indicators
  updateProgressBar();
  updateStepIndicators();
  updateNavigationButtons();

  // Re-animate first step
  gsap.set(".step.active", { opacity: 1, x: 0 });
  gsap.from(".step.active .step-header", {
    duration: 0.8,
    y: 50,
    opacity: 0,
    ease: "power3.out",
    delay: 0.2,
  });
}

// Handle browser back/forward buttons
window.addEventListener("popstate", function (e) {
  if (e.state && e.state.step) {
    currentStep = e.state.step;
    showStep(currentStep);
  }
});

// Add step to browser history
function addToHistory(step) {
  const url = new URL(window.location);
  url.searchParams.set("step", step);
  if (logoNameFromIndex) {
    url.searchParams.set("name", logoNameFromIndex);
  }
  history.pushState({ step: step }, "", url);
}

// Show specific step (for history navigation)
function showStep(stepNumber) {
  document.querySelectorAll(".step").forEach((step) => {
    step.classList.remove("active");
  });

  document.getElementById(`step${stepNumber}`).classList.add("active");
  updateProgressBar();
  updateStepIndicators();
  updateNavigationButtons();
}

// 🚗 Continuous smooth Slick slider (like car on road)
let sliderInitialized = false;

function initSlickSlider() {
  const $slider = $(".slick-slider");
  
  // Check if slider exists
  if ($slider.length === 0) {
    return;
  }
  
  // Prevent multiple initializations
  if (sliderInitialized && $slider.hasClass('slick-initialized')) {
    return;
  }
  
  // Destroy existing slider if it exists to prevent conflicts
  if ($slider.hasClass('slick-initialized')) {
    try {
      $slider.slick('unslick');
    } catch(e) {
      console.log('Slider already destroyed or not initialized');
    }
  }
  
  // Make sure slider is visible before initialization
  $slider.css({
    'visibility': 'visible',
    'opacity': '1',
    'display': 'block'
  });
  
  // Initialize immediately without delay to prevent glitch
  try {
    $slider.slick({
      slidesToShow: 7,
      slidesToScroll: 1,
      autoplay: true,
      autoplaySpeed: 0,
      speed: 6000,
      cssEase: "linear",
      infinite: true,
      arrows: false,
      dots: false,
      pauseOnHover: false,
      pauseOnFocus: false,
      swipe: false,
      touchMove: false,
      adaptiveHeight: false,
      fade: false,
      responsive: [
        {
          breakpoint: 1440,
          settings: {
            slidesToShow: 7,
          },
        },
        {
          breakpoint: 1024,
          settings: {
            slidesToShow: 5,
          },
        },
        {
          breakpoint: 768,
          settings: {
            slidesToShow: 4,
          },
        },
        {
          breakpoint: 480,
          settings: {
            slidesToShow: 2.5,
          },
        },
      ],
    });
    sliderInitialized = true;
    
    // Force visibility after initialization
    $slider.css({
      'visibility': 'visible',
      'opacity': '1',
      'display': 'block'
    });
  } catch(e) {
    console.error('Error initializing slider:', e);
  }
}

// Initialize immediately when DOM is ready
$(document).ready(function () {
  // Show slider container immediately
  $('.slider-container').css({
    'visibility': 'visible',
    'opacity': '1',
    'display': 'block'
  });
  
  // Initialize slider
  initSlickSlider();
});

// Also initialize after window load as fallback
$(window).on('load', function() {
  if (!sliderInitialized) {
    $('.slider-container').css({
      'visibility': 'visible',
      'opacity': '1',
      'display': 'block'
    });
    initSlickSlider();
  }
});